<?php

declare(strict_types=1);

use App\Constants\Status;
use App\Http\Middleware\Ability\CheckAbilities;
use App\Http\Middleware\Ability\UddoktaPayCheckAbilities;
use App\Http\Middleware\Ability\UddoktaPayVerifyCheckAbilities;
use App\Http\Middleware\Api\ValidateApi;
use App\Http\Middleware\Api\ValidateUddoktaPayApi;
use App\Http\Middleware\Api\ValidateUddoktaPayVerifyApi;
use App\Http\Middleware\SetLocale;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Configuration\Exceptions;
use Illuminate\Foundation\Configuration\Middleware;
use Illuminate\Http\Request;
use Sentry\Laravel\Integration;
use Symfony\Component\HttpFoundation\Response;

return Application::configure(basePath: dirname(__DIR__))
    ->withRouting(
        web: base_path('routes/web.php'),
        api: base_path('routes/api.php'),
        commands: base_path('routes/console.php'),
        health: '/uptime',
        apiPrefix: '',
    )
    ->withMiddleware(function (Middleware $middleware): void {
        $middleware->trustProxies(at: '*');

        $middleware->validateCsrfTokens(except: [
            'api/*',
            '*/api/*',
            'gateway/*',
            '*/gateway/*',
        ]);

        $middleware->alias([
            'set-locale' => SetLocale::class,
            'validate.api' => ValidateApi::class,
            'ability' => CheckAbilities::class,

            'validate.api.uddoktapay' => ValidateUddoktaPayApi::class,
            'validate.api.verify.uddoktapay' => ValidateUddoktaPayVerifyApi::class,

            'uddoktapay.ability' => UddoktaPayCheckAbilities::class,
            'uddoktapay.verify.ability' => UddoktaPayVerifyCheckAbilities::class,
        ]);
    })
    ->withExceptions(function (Exceptions $exceptions): void {
        if (app()->isProduction()) {
            try {
                Integration::handles($exceptions);
            } catch (Exception) {
                //  Do nothing if Sentry is not configured properly
            }
        }

        $exceptions->shouldRenderJsonWhen(fn (Request $request, Throwable $e): bool => $request->is('api/*') || $request->expectsJson());

        $exceptions->render(function (Request $request, AuthenticationException $e) {
            if ($request->is('api/*')) {
                return response()->json([
                    'status' => Status::ERROR,
                    'message' => 'Unauthenticated',
                ], 401);
            }
        });

        $exceptions->respond(function (Response $response) {
            $status = $response->getStatusCode();

            if (request()->is('api/*') || request()->expectsJson()) {
                return match ($status) {
                    Response::HTTP_FOUND => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'An unexpected redirect occurred',
                    ], Response::HTTP_FOUND),

                    Response::HTTP_UNAUTHORIZED => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Unauthorized request',
                    ], Response::HTTP_UNAUTHORIZED),

                    Response::HTTP_NOT_FOUND => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Not Found',
                    ], Response::HTTP_NOT_FOUND),

                    Response::HTTP_METHOD_NOT_ALLOWED => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Method not allowed',
                    ], Response::HTTP_METHOD_NOT_ALLOWED),

                    Response::HTTP_FORBIDDEN => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Forbidden access',
                    ], Response::HTTP_FORBIDDEN),

                    Response::HTTP_TOO_MANY_REQUESTS => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Too many requests',
                    ], Response::HTTP_TOO_MANY_REQUESTS),

                    Response::HTTP_INTERNAL_SERVER_ERROR => response()->json([
                        'status' => Status::ERROR,
                        'message' => 'Internal server error',
                    ], Response::HTTP_INTERNAL_SERVER_ERROR),

                    default => $response,
                };
            }

            return $response;
        });
    })->create();
