# Gatewayify

A comprehensive Laravel package for payment gateway integration with support for multiple providers including Stripe, PayPal, bKash, Nagad, SSLCommerz, Paddle, and more.

## Features

- **Multi-Gateway Support**: 20+ payment providers out of the box
- **Unified API**: Consistent interface across all payment gateways
- **Event-Driven**: Built-in events for payment lifecycle management
- **IPN Support**: Automatic payment verification via webhooks
- **Refund Capabilities**: Built-in refund support for compatible gateways
- **Validation**: Automatic gateway configuration validation
- **Extensible**: Easy to add custom payment providers
- **Laravel Integration**: Native Laravel service provider with route management

## Supported Payment Gateways

### Global Providers
- **Stripe** - Credit cards, digital wallets
- **PayPal** - PayPal payments and PayPal Personal
- **Paddle** - Subscription and one-time payments

### Bangladesh (Local)
- **bKash** - API (Checkout & Tokenized)
- **Nagad** - API integration
- **SSLCommerz** - Multiple methods (bKash, Nagad, Rocket, Upay, etc.)
- **Upay** - Direct API integration

### Others
- **Payeer** - Multi-currency wallet

## Installation

Install the package via Composer:

```bash
composer require revoltify/gatewayify
```

Publish the views (optional):

```bash
php artisan vendor:publish --tag=gatewayify-views
```

## Quick Start

### 1. Implement the Contracts

Your Gateway model must implement the `Gatewayify` contract:

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Revoltify\Gatewayify\Concerns\HasGatewayify;
use Revoltify\Gatewayify\Contracts\Gatewayify;

class Gateway extends Model implements Gatewayify
{
    use HasGatewayify;

    protected $fillable = ['name', 'code', 'currency', 'config_data', 'extra_data', 'is_active'];

    protected $casts = [
        'config_data' => 'array',
        'extra_data' => 'array',
        'is_active' => 'boolean',
    ];

    // Required method implementations
    public function gatewayifyReferenceId(): int|string
    {
        return $this->code;
    }

    public function gatewayifyDirectory(): string
    {
        return $this->code;
    }

    public function gatewayifyCurrency(): string
    {
        return $this->currency;
    }

    protected function getGatewayifyConfigColumn(): string
    {
        return 'config_data';
    }

    protected function getGatewayifyExtraColumn(): string
    {
        return 'extra_data';
    }

    // Route name - package automatically appends .success, .cancel, .ipn, .status
    protected function gatewayifyRouteName(): string
    {
        return 'gateway';
    }
}
```

Your Payment model must implement the `Paymentable` contract:

```php
<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Revoltify\Gatewayify\Contracts\Paymentable;
use Revoltify\Gatewayify\Contracts\Gatewayify;

class Payment extends Model implements Paymentable
{
    protected $fillable = [
        'payment_id', 'amount', 'currency', 'status',
        'gateway_transaction_id', 'gateway_reference_id'
    ];

    // Required method implementations
    public function paymentableReferenceId(): int|string
    {
        return $this->payment_id;
    }

    public function paymentableAmount(): string
    {
        return $this->amount;
    }

    public function paymentableIsAlreadyCompleted(): bool
    {
        return $this->status === 'completed';
    }

    public function paymentableGatewayReferenceId(): null|int|string
    {
        return $this->gateway_reference_id;
    }

    public function paymentableGatewayTransactionId(): int|string
    {
        return $this->gateway_transaction_id;
    }

    public function paymentableHasExistingTransaction(Gatewayify $gateway, string $transactionId): bool
    {
        return static::where('gateway_transaction_id', $transactionId)->exists();
    }

    public function scopeByReferenceId(Builder $query, string $referenceId)
    {
        return $query->where('payment_id', $referenceId);
    }

    // URL methods
    public function paymentableSuccessUrl(): string
    {
        return route('payment.success', $this->payment_id);
    }

    public function paymentableCancelUrl(): string
    {
        return route('payment.cancel', $this->payment_id);
    }

    public function paymentableIPNUrl(): ?string
    {
        return route('payment.ipn', $this->payment_id);
    }

    // Customer information
    public function paymentableCustomerName(): ?string
    {
        return $this->customer_name;
    }

    public function paymentableCustomerEmail(): ?string
    {
        return $this->customer_email;
    }

    public function paymentableCustomerPhone(): ?string
    {
        return $this->customer_phone;
    }

    public function paymentableProductName(): ?string
    {
        return 'Payment for Order #' . $this->payment_id;
    }
}
```

### 2. Set up Routes

```php
<?php

use App\Http\Controllers\Gateway\GatewayController;
use Revoltify\Gatewayify\Routing\GatewayifyRouteGenerator;

// Generate all gateway routes automatically
GatewayifyRouteGenerator::create(GatewayController::class)->generate();
```

### 3. Create Gateway Controller

```php
<?php

namespace App\Http\Controllers\Gateway;

use App\Models\Gateway;
use App\Models\Payment;
use Revoltify\Gatewayify\Http\Controllers\AbstractGatewayController;
use Revoltify\Gatewayify\Contracts\Gatewayify;
use Revoltify\Gatewayify\Contracts\Paymentable;

class GatewayController extends AbstractGatewayController
{
    protected function getGatewayModel(): string
    {
        return Gateway::class;
    }

    protected function getPaymentModel(): string
    {
        return Payment::class;
    }

    protected function findGateway(string $gatewayReference): Gatewayify
    {
        return Gateway::where('code', $gatewayReference)->firstOrFail();
    }

    protected function findPayment(string $paymentReference): Paymentable
    {
        return Payment::where('payment_id', $paymentReference)->firstOrFail();
    }
}
```

### 4. Database Migrations

Create the gateways table:

```php
Schema::create('gateways', function (Blueprint $table) {
    $table->id();
    $table->string('name');
    $table->string('code')->unique();
    $table->string('currency', 3);
    $table->json('config_data')->nullable();
    $table->json('extra_data')->nullable();
    $table->boolean('is_active')->default(true);
    $table->timestamps();
});
```

Update your payments table:

```php
Schema::table('payments', function (Blueprint $table) {
    $table->string('payment_id')->unique();
    $table->decimal('amount', 15, 2);
    $table->string('currency', 3);
    $table->string('status')->default('pending');
    $table->string('gateway_transaction_id')->nullable();
    $table->string('gateway_reference_id')->nullable();
    $table->string('customer_name')->nullable();
    $table->string('customer_email')->nullable();
    $table->string('customer_phone')->nullable();
});
```

## Usage

### Process a Payment

```php
use App\Models\Gateway;
use App\Models\Payment;
use Revoltify\Gatewayify\Services\GatewayifyService;

// Find gateway and payment
$gateway = Gateway::where('code', 'stripe')->first();
$payment = Payment::where('payment_id', 'PAY123')->first();

// Process checkout
$gatewayService = new GatewayifyService();
$response = $gatewayService->checkout($gateway, $payment);

// Handle response
return $response->toResponse(request());
```

### Listen to Payment Events

```php
<?php

namespace App\Listeners;

use Revoltify\Gatewayify\Events\PaymentCompleted;

class PaymentCompletedListener
{
    public function handle(PaymentCompleted $event): void
    {
        $payment = $event->payment;
        $gateway = $event->gateway;
        $transactionId = $event->transactionId;
        
        // Update payment status
        $payment->update([
            'status' => 'completed',
            'gateway_transaction_id' => $transactionId,
            'gateway_reference_id' => $event->gatewayReferenceId,
        ]);
        
        // Send notification email
        // Update inventory
        // Generate invoice
    }
}
```

Register the listener in `EventServiceProvider`:

```php
protected $listen = [
    \Revoltify\Gatewayify\Events\PaymentCompleted::class => [
        \App\Listeners\PaymentCompletedListener::class,
    ],
    \Revoltify\Gatewayify\Events\PaymentPending::class => [
        \App\Listeners\PaymentPendingListener::class,
    ],
];
```

### Process Refunds

```php
use Revoltify\Gatewayify\Contracts\RefundCapable;

$gateway = Gateway::where('code', 'stripe')->first();
$payment = Payment::where('payment_id', 'PAY123')->first();

$provider = $gateway->gatewayify();

if ($provider instanceof RefundCapable) {
    $refundId = $provider->refund(
        payment: $payment,
        amount: 50.00,
        reason: 'Customer requested refund'
    );
    
    echo "Refund processed: {$refundId}";
}
```

### Gateway Configuration

```php
// Get gateway configuration fields
$gateway = Gateway::where('code', 'stripe')->first();
$config = $gateway->gatewayifyConfig();

foreach ($config->getFields() as $field) {
    echo $field->getTitle() . ': ' . $field->getType();
}

// Validate gateway configuration
try {
    $gateway->validateGatewayify([
        'secret_key' => 'sk_test_...',
    ]);
    echo 'Configuration is valid';
} catch (\Exception $e) {
    echo 'Invalid configuration: ' . $e->getMessage();
}
```

## Creating Custom Payment Providers

### 1. Create Provider Class

```php
<?php

namespace App\Modules\Gateways\CustomGateway;

use Illuminate\Http\Request;
use Revoltify\Gatewayify\BaseGatewayProvider;
use Revoltify\Gatewayify\Config\GatewayMetadata;
use Revoltify\Gatewayify\Contracts\Initializable;
use Revoltify\Gatewayify\Contracts\Paymentable;
use Revoltify\Gatewayify\Response\GatewayResponse;
use Revoltify\Support\Forms\Components\Field;
use Revoltify\Support\Forms\Form;

class ProcessPayment extends BaseGatewayProvider implements Initializable
{
    public function init(): void
    {
        // Initialize your gateway client
        // Validate required configuration
    }

    public static function metadata(): GatewayMetadata
    {
        return GatewayMetadata::make()
            ->name('Custom Gateway')
            ->group('custom')
            ->currency('USD', '$');
    }

    public function config(): Form
    {
        return Form::make([
            Field::make('api_key')
                ->label('API Key')
                ->required(),
            
            Field::make('secret_key')
                ->label('Secret Key')
                ->password()
                ->required(),
        ]);
    }

    public function checkout(Paymentable $payment): GatewayResponse
    {
        // Create payment with your gateway
        $paymentUrl = 'https://gateway.com/pay/xyz';
        
        return GatewayResponse::redirect($paymentUrl);
    }

    public function success(Request $request, Paymentable $payment): GatewayResponse
    {
        // Verify payment with your gateway
        if ($this->verifyPayment($request)) {
            $this->markAsCompleted($payment, $request->get('transaction_id'));
            return $this->redirectToSuccessURL($payment);
        }
        
        return GatewayResponse::error('Payment verification failed');
    }
}
```

### 2. Register Provider

Place your provider in one of these locations:
- `app/Modules/Gateways/CustomGateway/ProcessPayment.php` (higher priority)
- Package providers are auto-discovered

## Available Events

| Event | Description |
|-------|-------------|
| `PaymentCompleted` | Fired when payment is successfully completed |
| `PaymentPending` | Fired when payment is in pending state |
| `PaymentCompletionFailed` | Fired when payment completion fails |
| `PaymentPendingFailed` | Fired when payment pending processing fails |

## Configuration Fields

The package provides a fluent interface for defining gateway configuration fields:

```php
Field::make('api_key')
    ->label('API Key')
    ->text()
    ->required()
    ->helperText('Get your API key from the gateway dashboard'),

Field::make('sandbox')
    ->select(['enable' => 'Enable', 'disable' => 'Disable'])
    ->default('disable'),

Field::make('webhook_secret')
    ->password()
    ->revealable()
    ->columnSpanFull(),
```

## Route Customization

Customize route generation with a single prefix - the package automatically handles all route variations:

```php
GatewayifyRouteGenerator::create(GatewayController::class)
    ->prefix('payments')           // Base prefix
    ->name('payments')            // Base route name
    ->middleware(['auth', 'verified'])
    ->generate();
```

With the above configuration, your Gateway model only needs to specify the base route name:

```php
// In your Gateway model - just specify the base route name
protected function gatewayifyRouteName(): string
{
    return 'payments';    // Package auto-generates route names:
                         // payments.success, payments.cancel, 
                         // payments.ipn, payments.status
}
```

**That's it!** The trait automatically appends `.success`, `.cancel`, `.ipn`, and `.status` to your route name.

Generated routes:
- `GET /payments/{gateway}/checkout/{payment}` → `payments.checkout`
- `POST /payments/{gateway}/ipn/{payment?}` → `payments.ipn`
- `GET|POST /payments/{gateway}/success/{payment?}` → `payments.success`
- `GET|POST /payments/{gateway}/cancel/{payment?}` → `payments.cancel`  
- `GET /payments/{gateway}/status/{payment}` → `payments.status`

## Error Handling

```php
use Revoltify\Gatewayify\Exceptions\GatewayException;

try {
    $response = $gatewayService->checkout($gateway, $payment);
} catch (GatewayException $e) {
    // Handle gateway-specific errors
    logger()->error('Gateway error: ' . $e->getMessage(), $e->getContext());
}
```


## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

## License

The MIT License (MIT). Please see [License File](LICENSE) for more information.

## Credits

- [Revoltify Team](https://github.com/revoltify)
- [All Contributors](../../contributors)
---

Made with ❤️ by [Revoltify](https://github.com/revoltify)