<?php

declare(strict_types=1);

use Revoltify\Mfs\Enums\MfsMethodCodes;
use Revoltify\Mfs\TransactionParser;

beforeEach(function () {
    $this->service = new TransactionParser;
});

test('bkash personal send money with ref', function () {
    $sms = 'You have received Tk 10,000.00 from 01111111111.Ref SomeRef. Fee Tk 0.00. Balance Tk 20,000.00. TrxID BG39RL1234 at 14/10/2024 17:02';

    $expected = [
        'provider' => 'bkash',
        'type' => 'personal',
        'transaction_id' => 'BG39RL1234',
        'amount' => '10,000.00',
        'phone_number' => '01111111111',
        'balance' => '20,000.00',
        'ref' => 'SomeRef',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash personal send money without ref', function () {
    $sms = 'You have received Tk 10,000.00 from 01111111111. Fee Tk 0.00. Balance Tk 20,000.00. TrxID BG39RL1234 at 14/10/2024 17:02';
    $expected = [
        'provider' => 'bkash',
        'type' => 'personal',
        'transaction_id' => 'BG39RL1234',
        'amount' => '10,000.00',
        'phone_number' => '01111111111',
        'balance' => '20,000.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash personal bank deposit', function () {
    $sms = 'You have received deposit from iBanking of Tk 2,000.00 from Islami Bank. Fee Tk 0.00. Balance Tk 4,000.00. TrxID BE747Y1234 at 14/10/2024 20:24';
    $expected = [
        'provider' => 'bkash',
        'type' => 'personal',
        'transaction_id' => 'BE747Y1234',
        'amount' => '2,000.00',
        'phone_number' => 'Islami Bank',
        'balance' => '4,000.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash personal cash in', function () {
    $sms = 'Cash In Tk 1000.00 from 01111111111 successful. Fee Tk 0.00. Balance Tk 2000.00. TrxID BDN1S12345 at 14/10/2024 10:19';
    $expected = [
        'provider' => 'bkash',
        'type' => 'personal',
        'transaction_id' => 'BDN1S12345',
        'amount' => '1000.00',
        'phone_number' => '01111111111',
        'balance' => '2000.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash merchant with ref', function () {
    $sms = 'You have received payment Tk 9.00 from 01111111111. Ref Test. Fee Tk 0.00. Balance Tk 11.78. TrxID BJE9FW1234 at 14/10/2024 12:13';
    $expected = [
        'provider' => 'bkash',
        'type' => 'merchant',
        'transaction_id' => 'BJE9FW1234',
        'amount' => '9.00',
        'phone_number' => '01111111111',
        'balance' => '11.78',
        'ref' => 'Test',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash merchant without ref', function () {
    $sms = 'You have received payment Tk 255.00 from 01111111111. Fee Tk 0.00. Balance Tk 1,010.36. TrxID BG33R61234 at 14/10/2024 09:39';
    $expected = [
        'provider' => 'bkash',
        'type' => 'merchant',
        'transaction_id' => 'BG33R61234',
        'amount' => '255.00',
        'phone_number' => '01111111111',
        'balance' => '1,010.36',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash merchant to merchant with ref', function () {
    $sms = 'You have received Tk 10.00 from Uddoktapay_01111111111 01111111111. Ref SomeRef. Fee Tk 0.00. Balance Tk 100.60. TrxID BJE9FW1234 at 14/10/2024 12:10';
    $expected = [
        'provider' => 'bkash',
        'type' => 'merchant',
        'transaction_id' => 'BJE9FW1234',
        'amount' => '10.00',
        'phone_number' => '01111111111',
        'balance' => '100.60',
        'ref' => 'SomeRef',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash merchant to merchant without ref', function () {
    $sms = 'You have received Tk 10.00 from Uddoktapay_01111111111 01111111111. Ref . Fee Tk 0.00. Balance Tk 100.60. TrxID BJE9FW1234 at 14/10/2024 12:10';
    $expected = [
        'provider' => 'bkash',
        'type' => 'merchant',
        'transaction_id' => 'BJE9FW1234',
        'amount' => '10.00',
        'phone_number' => '01111111111',
        'balance' => '100.60',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('bkash agent cash in', function () {
    $sms = 'Cash Out Tk 1,600.00 from 0111XXXX111 successful. Fee Tk 0.00. Balance Tk 2,000.64. TrxID BJC2E01234 at 14/10/2024 00:18';
    $expected = [
        'provider' => 'bkash',
        'type' => 'agent',
        'transaction_id' => 'BJC2E01234',
        'amount' => '1,600.00',
        'phone_number' => '0111XXXX111',
        'balance' => '2,000.64',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::BKASH, $sms);
    expect($parsedData)->toEqual($expected);
});

test('nagad personal send money', function () {
    $sms = 'Money Received. Amount: Tk 200.00 Sender: 01111111111 Ref: SomeRef TxnID: 73AF1234 Balance: Tk 242.43 14/10/2024 13:45';

    $expected = [
        'provider' => 'nagad',
        'type' => 'personal',
        'transaction_id' => '73AF1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => 'SomeRef',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::NAGAD, $sms);
    expect($parsedData)->toEqual($expected);
});

test('nagad personal cash in', function () {
    $sms = 'Cash In Received. Amount: Tk 100.00 Uddokta: 01111111111 TxnID: 73AH1234 Balance: 342.43 14/10/2024 22:00';

    $expected = [
        'provider' => 'nagad',
        'type' => 'personal',
        'transaction_id' => '73AH1234',
        'amount' => '100.00',
        'phone_number' => '01111111111',
        'balance' => '342.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::NAGAD, $sms);
    expect($parsedData)->toEqual($expected);
});

test('nagad personal bank deposit', function () {
    $sms = 'Add Money Successful. From: VISA-EASTERN BANK LIMITED/VISA Amount: Tk 50 Sender: 01111111111 TxnID: 73AL1234 Balance: Tk 290.93 16/10/2024 11:48';

    $expected = [
        'provider' => 'nagad',
        'type' => 'personal',
        'transaction_id' => '73AL1234',
        'amount' => '50',
        'phone_number' => '01111111111',
        'balance' => '290.93',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::NAGAD, $sms);
    expect($parsedData)->toEqual($expected);
});

test('nagad merchant', function () {
    $sms = 'Payment Received. Amount: Tk 200.00 Customer: 01111111111 TxnID: 73AF1234 Balance: Tk 242.43 14/10/2024 13:45';

    $expected = [
        'provider' => 'nagad',
        'type' => 'merchant',
        'transaction_id' => '73AF1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::NAGAD, $sms);
    expect($parsedData)->toEqual($expected);
});

test('nagad agent', function () {
    $sms = 'Cash Out Received. Amount: Tk 198.00 Customer: 01111111111 TxnID: 73A41234 Comm: Tk 0.81 Balance: Tk 242.43 10/10/2024 22:39';

    $expected = [
        'provider' => 'nagad',
        'type' => 'agent',
        'transaction_id' => '73A41234',
        'amount' => '198.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::NAGAD, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket personal send money', function () {
    $sms = 'Tk150.00 received from A/C:***714 Fee:Tk0, Your A/C Balance: Tk2,000.51 TxnId:5143728891 Date:08-FEB-25 06:24:34 pm. Download https://bit.ly/nexuspay';

    $expected = [
        'provider' => 'rocket',
        'type' => 'personal',
        'transaction_id' => '5143728891',
        'amount' => '150.00',
        'phone_number' => '***714',
        'balance' => '2,000.51',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket personal cash in', function () {
    $sms = 'Cash-In from A/C: ***714 Tk100.00 Fee: Tk.00, Your A/C Balance: Tk290.31.TxnId:4871641234 Date:14-OCT-24 10:11:44 pm. Download https://bit.ly/nexuspay';

    $expected = [
        'provider' => 'rocket',
        'type' => 'personal',
        'transaction_id' => '4871641234',
        'amount' => '100.00',
        'phone_number' => '***714',
        'balance' => '290.31',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket personal bank deposit', function () {
    $sms = 'Tk10.00 credited from card ***0899. Fee:Tk.00 NetBal:Tk178.64 TxnId:4874564686 Date:16-OCT-24 11:32:52 am.';

    $expected = [
        'provider' => 'rocket',
        'type' => 'personal',
        'transaction_id' => '4874564686',
        'amount' => '10.00',
        'phone_number' => 'card ***0899',
        'balance' => '178.64',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket personal nexus pay deposit', function () {
    $sms = 'Tk450.00 transferred to A/C:***879 Fee:Tk.00, Your A/C Balance: Tk.00 TxnId:5397026295 Date:14-MAY-25 05:19:49 pm.';

    $expected = [
        'provider' => 'rocket',
        'type' => 'personal',
        'transaction_id' => '5397026295',
        'amount' => '450.00',
        'phone_number' => '***879',
        'balance' => '00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket merchant', function () {
    $sms = 'Tk.10.00 received from A/C:***879 Ref No: Test Ref Fee: Tk.00 NetBal: Tk29.86 TxnId: 4874581234 Date:16-OCT-24 11:11:51 am.';

    $expected = [
        'provider' => 'rocket',
        'type' => 'merchant',
        'transaction_id' => '4874581234',
        'amount' => '10.00',
        'phone_number' => '***879',
        'balance' => '29.86',
        'ref' => 'Test Ref',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('rocket agent', function () {
    $sms = 'B2C: Cash-Out from A/C: ***932 Tk51.00 Comm:Tk.21; A/C Balance: Tk16,450.35.TxnId: 5751795462 Date:23-SEP-25 10:14:32 am. Download https://bit.ly/nexuspay';

    $expected = [
        'provider' => 'rocket',
        'type' => 'agent',
        'transaction_id' => '5751795462',
        'amount' => '51.00',
        'phone_number' => '***932',
        'balance' => '16,450.35',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::ROCKET, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay personal send money with ref', function () {
    $sms = 'Tk. 200.00 has been received from 01111111111. Ref-Test Ref. Balance Tk. 300.06. TrxID 01J7BE1234 at 09/09/2024 19:02.';

    $expected = [
        'provider' => 'upay',
        'type' => 'personal',
        'transaction_id' => '01J7BE1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '300.06',
        'ref' => 'Test Ref',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay personal send money without ref', function () {
    $sms = 'Tk. 200.00 has been received from 01111111111. Ref-. Balance Tk. 300.06. TrxID 01J7BE1234 at 09/09/2024 19:02.';

    $expected = [
        'provider' => 'upay',
        'type' => 'personal',
        'transaction_id' => '01J7BE1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '300.06',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay personal cash in', function () {
    $sms = 'Cash-In of Tk. 100.00 from 01111111111 is successful. Balance Tk. 100.79. TrxID 01JA5W1234 at 14/10/2024 22:12.';

    $expected = [
        'provider' => 'upay',
        'type' => 'personal',
        'transaction_id' => '01JA5W1234',
        'amount' => '100.00',
        'phone_number' => '01111111111',
        'balance' => '100.79',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay personal bank deposit with ref', function () {
    $sms = 'Tk.100.00 has been received from Debit/Credit Card. Balance Tk.200.79. Ref-Test Ref. TrxID 01JA5X1234 at 14/10/2024 22:17';

    $expected = [
        'provider' => 'upay',
        'type' => 'personal',
        'transaction_id' => '01JA5X1234',
        'amount' => '100.00',
        'phone_number' => 'Debit/Credit Card',
        'balance' => '200.79',
        'ref' => 'Test Ref',
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay personal bank deposit without ref', function () {
    $sms = 'Tk.100.00 has been received from Debit/Credit Card. Balance Tk.200.79. Ref-. TrxID 01JA5X1234 at 14/10/2024 22:17';

    $expected = [
        'provider' => 'upay',
        'type' => 'personal',
        'transaction_id' => '01JA5X1234',
        'amount' => '100.00',
        'phone_number' => 'Debit/Credit Card',
        'balance' => '200.79',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('upay merchant', function () {
    $sms = '';

    $expected = [
        'provider' => 'upay',
        'type' => 'merchant',
        'transaction_id' => '73AF1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('upay agent', function () {
    $sms = 'You have received Cash-out of Tk. 100.00 from 01111111111. Comm: TK. 0.4095. Balance Tk. 100.71. TrxID 01J7P31234 at 13/09/2024 22:37.';

    $expected = [
        'provider' => 'upay',
        'type' => 'agent',
        'transaction_id' => '01J7P31234',
        'amount' => '100.00',
        'phone_number' => '01111111111',
        'balance' => '100.71',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::UPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('cellfin personal send money', function () {
    $sms = 'Islami Bank CellFin Received 10.00 Tk From CellFin: 01111111111 To CellFin: 02222222222 TrxId: 8024101600001234 @16/10/2024 12:00 PM';

    $expected = [
        'provider' => 'cellfin',
        'type' => 'personal',
        'transaction_id' => '8024101600001234',
        'amount' => '10.00',
        'phone_number' => '01111111111',
        'balance' => null,
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::CELLFIN, $sms);
    expect($parsedData)->toEqual($expected);
});

test('cellfin personal from bank', function () {
    $sms = 'Islami Bank CellFin Received: 3,550.00 From: 20503550212345678 To CellFin: 01111111111 TrxId: 3424101000031234 @10/10/2024 01:19 PM';

    $expected = [
        'provider' => 'cellfin',
        'type' => 'personal',
        'transaction_id' => '3424101000031234',
        'amount' => '3,550.00',
        'phone_number' => '20503550212345678',
        'balance' => null,
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::CELLFIN, $sms);
    expect($parsedData)->toEqual($expected);
});

test('tap personal send money', function () {
    $sms = 'Received Tk 10.00 from 8801111111111. Balance Tk. 10.00. TxID: PAYOY0R9N54PB63. Call 16733';

    $expected = [
        'provider' => 'tap',
        'type' => 'personal',
        'transaction_id' => 'PAYOY0R9N54PB63',
        'amount' => '10.00',
        'phone_number' => '8801111111111',
        'balance' => '10.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::TAP, $sms);
    expect($parsedData)->toEqual($expected);
});

test('tap personal cash in', function () {
    $sms = '';

    $expected = [
        'provider' => 'tap',
        'type' => 'personal',
        'transaction_id' => '4871641234',
        'amount' => '100.00',
        'phone_number' => '01111111111',
        'balance' => '290.31',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::TAP, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('tap personal bank deposit', function () {
    $sms = 'Received Tk 100.00 from VISA Card. Fee Tk 0.00. Balance Tk 101.00. TxID: CARDOY0R9M3WNCFT at 11/03/2024 18:08:19';

    $expected = [
        'provider' => 'tap',
        'type' => 'personal',
        'transaction_id' => 'CARDOY0R9M3WNCFT',
        'amount' => '100.00',
        'phone_number' => 'VISA Card',
        'balance' => '101.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::TAP, $sms);
    expect($parsedData)->toEqual($expected);
});

test('tap merchant', function () {
    $sms = '';

    $expected = [
        'provider' => 'tap',
        'type' => 'merchant',
        'transaction_id' => '73AF1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::TAP, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('tap agent', function () {
    $sms = '';

    $expected = [
        'provider' => 'tap',
        'type' => 'agent',
        'transaction_id' => '4772511234',
        'amount' => '980.00',
        'phone_number' => '01111111111',
        'balance' => '1448.71',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::TAP, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('okwallet personal send money with ref', function () {
    $sms = '(OK Wallet) Successfully received Tk 11 from A/C 01111111111. Ref: Test Ref. Balance Tk 22. TrxID 72411037395187 at 03/11/24 18:29';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'personal',
        'transaction_id' => '72411037395187',
        'amount' => '11',
        'phone_number' => '01111111111',
        'balance' => '22',
        'ref' => 'Test Ref',
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
});

test('okwallet personal send money without ref', function () {
    $sms = '(OK Wallet) Successfully received Tk 11 from A/C 01111111111. Ref: . Balance Tk 22. TrxID 72411037395187 at 03/11/24 18:29';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'personal',
        'transaction_id' => '72411037395187',
        'amount' => '11',
        'phone_number' => '01111111111',
        'balance' => '22',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
});

test('okwallet personal cash in', function () {
    $sms = '';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'personal',
        'transaction_id' => '4871641234',
        'amount' => '100.00',
        'phone_number' => '01111111111',
        'balance' => '290.31',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('okwallet personal bank deposit', function () {
    $sms = '(OK Wallet) Successfully added money Tk 500 by VISA Card. Fee Tk 0. Balance Tk 514. TrxID 72411037395177 at 03/11/24 18:25';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'personal',
        'transaction_id' => '72411037395177',
        'amount' => '500',
        'phone_number' => 'VISA Card',
        'balance' => '514',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
});

test('okwallet merchant', function () {
    $sms = '';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'merchant',
        'transaction_id' => '73AF1234',
        'amount' => '200.00',
        'phone_number' => '01111111111',
        'balance' => '242.43',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('okwallet agent', function () {
    $sms = '';

    $expected = [
        'provider' => 'okwallet',
        'type' => 'agent',
        'transaction_id' => '4772511234',
        'amount' => '980.00',
        'phone_number' => '01111111111',
        'balance' => '1448.71',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(16269, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('pathaopay personal send money', function () {
    $sms = 'You have received Tk 10.00 from 8801111111111. Balance Tk 51.00 TrxID D1V77JO2G3 23:07 21/07/2025';

    $expected = [
        'provider' => 'pathaopay',
        'type' => 'personal',
        'transaction_id' => 'D1V77JO2G3',
        'amount' => '10.00',
        'phone_number' => '8801111111111',
        'balance' => '51.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::PATHAOPAY, $sms);
    expect($parsedData)->toEqual($expected);
});

test('pathaopay personal cash in', function () {
    $sms = '';

    $expected = [
        'provider' => 'pathaopay',
        'type' => 'personal',
        'transaction_id' => 'D1V77JO2G3',
        'amount' => '10.00',
        'phone_number' => '8801111111111',
        'balance' => '51.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::PATHAOPAY, $sms);
    expect($parsedData)->toEqual($expected);
})->todo();

test('pathaopay personal from card', function () {
    $sms = 'You have successfully added fund from card 4111 33** ** 0000. Amount Tk 50.00 Fee Tk 0 Balance Tk 50.00 TrxID D1V6Q20CID 22:37 21/07/2025';

    $expected = [
        'provider' => 'pathaopay',
        'type' => 'personal',
        'transaction_id' => 'D1V6Q20CID',
        'amount' => '50.00',
        'phone_number' => 'card 4111 33** ** 0000',
        'balance' => '50.00',
        'ref' => null,
    ];

    $parsedData = $this->service->parse(MfsMethodCodes::PATHAOPAY, $sms);
    expect($parsedData)->toEqual($expected);
});
